import os, sys
from pysmt.shortcuts import *
from pysmt.smtlib.parser import SmtLibParser
from pysmt.rewritings import *
import pysmt.operators as op
import argparse
import time


def dnf(formula):
    f = nnf(formula)
    def dnf_rec(f):
        if f.is_and():
            for i, a in enumerate(f.args()):
                if a.is_or():
                    rest = [aa for j, aa in enumerate(f.args()) if i != j]
                    l = [And(d, *rest) for d in a.args()]
                    return Or(*[dnf_rec(a) for a in l])
        return f
    ret = dnf_rec(f)
    return ret


class LinearIneq(object):
    def __init__(self, strict):
        self.strict = strict
        self.coeff = Real(0)
        self.vars = {}

    def add(self, coeff, var=None):
        if var is None:
            self.coeff = Real(self.coeff.constant_value() +
                              coeff.constant_value())
        else:
            c = self.vars.get(var, Real(0))
            self.vars[var] = Real(c.constant_value() + coeff.constant_value())


def coeff(linear, var):
    return linear.vars.get(var, Real(0)).constant_value()


def combine(l1, l2, var):
    res = LinearIneq(l1.strict or l2.strict)
    c1 = coeff(l1, var)
    c2 = coeff(l2, var)
    f1 = abs(c2 / c1)
    f2 = abs(c1 / c2)
    assert c1 != 0 and c2 != 0
    for v in l1.vars:
        if v != var:
            res.add(Real(f1 * coeff(l1, v)), v)
    for v in l2.vars:
        if v != var:
            res.add(Real(f2 * coeff(l2, v)), v)
    res.add(Real(f1 * l1.coeff.constant_value()))
    res.add(Real(f2 * l2.coeff.constant_value()))
    return res


def scale(linear, n):
    assert n > 0
    ret = LinearIneq(linear.strict)
    ret.add(Real(linear.coeff.constant_value() * n))
    for v in linear.vars:
        ret.add(Real(coeff(linear, v) * n), v)
    return ret


def to_linear(term, negate):
    assert term.node_type() in (op.LT, op.LE), term
    a, b = term.args()
    if negate:
        a, b = b, a
    strict = term.node_type() == (op.LT if not negate else op.LE)
    res = LinearIneq(strict)
    to_process = [(-1, a), (1, b)]
    while to_process:
        c, t = to_process[-1]
        to_process.pop()
        if t.is_times():
            for i, a in enumerate(t.args()):
                if a.is_constant():
                    to_process += [(c * a.constant_value(), aa)
                                   for j, aa in enumerate(t.args()) if i != j]
                    break
        elif t.is_plus():
            to_process += [(c, a) for a in t.args()]
        elif t.is_minus():
            if len(t.args()) == 1:
                to_process.append((-c, t.arg(0)))
            else:
                to_process.append((c, t.arg(0)))
                to_process += ([(-c, a) for a in t.args()[1:]])
        elif t.is_constant():
            res.add(Real(t.constant_value() * c))
        elif t.is_symbol():
            res.add(Real(c), t)
    return res


def to_term(linear, rhs_only=False):
    rhs = Plus(linear.coeff,
               *[Times(c, v) for (v, c) in linear.vars.iteritems()])
    if rhs_only:
        return rhs
    if linear.strict:
        return LT(Real(0), rhs)
    else:
        return LE(Real(0), rhs)


def get_atoms(formula):
    seen = set()
    to_process = [formula]
    while to_process:
        cur = to_process[-1]
        to_process.pop()
        if cur in seen:
            continue
        seen.add(cur)
        if cur.is_theory_relation():
            yield cur
        else:
            to_process += cur.args()
