from __future__ import print_function
from model import *

class Unroller(object):
    def __init__(self, model):
        self.model = model
        self.var2time = {}

    def get_time_var(self, x, k):
        if (x, k) not in self.var2time:
            v = FreshSymbol(x.get_type())
            self.var2time[(x, k)] = v
        return self.var2time[(x, k)]

    def at_time(self, formula, k):
        subst = {}
        for (xcur, xnext) in self.model.statevars:
            subst[xcur] = self.get_time_var(xcur, k)
            subst[xnext] = self.get_time_var(xcur, k+1)
        for x in formula.get_free_variables():
            if x not in subst:
                subst[x] = self.get_time_var(x, k)
        return formula.substitute(subst)

# end of class Unroller


class BMC(object):
    def __init__(self, model):
        self.model = model
        self.unroller = Unroller(model)
        self.solver = Solver()
        self.cex = None
        
    def check(self, max_k):
        bad = Not(self.model.invarprops[0])
        self.solver.add_assertion(self.unroller.at_time(self.model.init, 0))
        k = 0
        while True:
            if max_k is not None and k == max_k:
                break
            print('checking step %d' % k)
            badk = self.unroller.at_time(bad, k)
            self.solver.push()
            self.solver.add_assertion(badk)
            sat = self.solver.solve()
            if sat:
                self.build_cex(k)
                return False
            else:
                self.solver.pop()
                self.solver.add_assertion(Not(badk))
                self.solver.add_assertion(
                    self.unroller.at_time(self.model.trans, k))
            k += 1
        return None

    def build_cex(self, k):
        self.cex = []
        svars = set(v[0] for v in self.model.statevars)
        nvars = set(v[1] for v in self.model.statevars)
        allvars = sorted((self.model.init.get_free_variables() |
                          self.model.trans.get_free_variables() |
                          self.model.invarprops[0].get_free_variables()) - \
                         nvars,
                         key=lambda v : (v not in svars, str(v)))
        m = self.solver.get_model()
        for i in xrange(k):
            step = []
            for v in allvars:
                step.append((v, m[self.unroller.at_time(v, k)]))
            self.cex.append(step)

# end of class BMC


if __name__ == '__main__':
    import sys
    model = Model.read(sys.stdin)
    bmc = BMC(model)
    res = bmc.check(None)
    if res == False:
        print('found counterexample of length %d' % len(bmc.cex))
        for i, step in enumerate(bmc.cex):
            print('-- step %d --' % i)
            for (x, v) in step:
                print('  %s := %s' % (x, v))
        print('unsafe')
    else:
        print('unknown')
